#include <d3d9.h>
#include <d3dx9.h>

#include <algorithm>

#include "HomeworkRendererFunctions.h"
#include "Utilities.h"

struct EdgePoint
{
    float x, z, oow, fog;
    VertexData data;
};

unsigned int RasterizeScanLine(unsigned int y, EdgePoint const& e0, EdgePoint const& e1, RasterizeParams const& params, Pixel* pixels, unsigned int pixelsSize)
{
    int const x0 = max((int)floor(e0.x + 0.5f)    , (int)params.scissorX);
    int const x1 = min((int)floor(e1.x + 0.5f) - 1, (int)(params.scissorX + params.scissorWidth - 1));

    float const oodx = 65536.0f / (e1.x - e0.x);

    unsigned int resultCount = 0;

    int const dz   = int((e1.z               - e0.z              ) * oodx);
    int const doow = int((e1.oow             - e0.oow            ) * oodx);
    int const dfog = int((e1.fog             - e0.fog            ) * oodx);
    int const dr   = int((e1.data.diffuse.r  - e0.data.diffuse.r ) * oodx);
    int const dg   = int((e1.data.diffuse.g  - e0.data.diffuse.g ) * oodx);
    int const db   = int((e1.data.diffuse.b  - e0.data.diffuse.b ) * oodx);
    int const da   = int((e1.data.diffuse.a  - e0.data.diffuse.a ) * oodx);
    int const dsr  = int((e1.data.specular.r - e0.data.specular.r) * oodx);
    int const dsg  = int((e1.data.specular.g - e0.data.specular.g) * oodx);
    int const dsb  = int((e1.data.specular.b - e0.data.specular.b) * oodx);
    int const dsa  = int((e1.data.specular.a - e0.data.specular.a) * oodx);
    int const dtx  = int((e1.data.texCoord.x - e0.data.texCoord.x) * oodx);
    int const dty  = int((e1.data.texCoord.y - e0.data.texCoord.y) * oodx);

    float const t = float(x0) + 0.5f - e0.x;

    int z   = int(65536.0f * e0.z               + dz   * t);
    int oow = int(65536.0f * e0.oow             + doow * t);
    int fog = int(65536.0f * e0.fog             + dfog * t);
    int r   = int(65536.0f * e0.data.diffuse.r  + dr   * t);
    int g   = int(65536.0f * e0.data.diffuse.g  + dg   * t);
    int b   = int(65536.0f * e0.data.diffuse.b  + db   * t);
    int a   = int(65536.0f * e0.data.diffuse.a  + da   * t);
    int sr  = int(65536.0f * e0.data.specular.r + dsr  * t);
    int sg  = int(65536.0f * e0.data.specular.g + dsg  * t);
    int sb  = int(65536.0f * e0.data.specular.b + dsb  * t);
    int sa  = int(65536.0f * e0.data.specular.a + dsa  * t);
    int tx  = int(65536.0f * e0.data.texCoord.x + dtx  * t);
    int ty  = int(65536.0f * e0.data.texCoord.y + dty  * t);

    for (int x = x0; x <= x1; ++x)
    {
        if (resultCount >= pixelsSize)
        {
            throw "Rasterize: insufficient space for the result";
        }

        float const w = 1.0f / oow;

        Pixel& p = pixels[resultCount];

        p.x = x;
        p.y = y;
        p.z = z * (1 / 65536.0f);
        p.fog             = fog * w;
        p.data.diffuse.r  = r   * w;
        p.data.diffuse.g  = g   * w;
        p.data.diffuse.b  = b   * w;
        p.data.diffuse.a  = a   * w;
        p.data.specular.r = sr  * w;
        p.data.specular.g = sg  * w;
        p.data.specular.b = sb  * w;
        p.data.specular.a = sa  * w;
        p.data.texCoord.x = tx  * w;
        p.data.texCoord.y = ty  * w;

        ++resultCount;

        z   += dz  ;
        oow += doow;
        fog += dfog;
        r   += dr  ;
        g   += dg  ;
        b   += db  ;
        a   += da  ;
        sr  += dsr ;
        sg  += dsg ;
        sb  += dsb ;
        sa  += dsa ;
        tx  += dtx ;
        ty  += dty ;
    }

    return resultCount;
}

EdgePoint InterpolateEdgePoint(EdgePoint const& e0, EdgePoint const& e1, float t, float t1)
{
    EdgePoint  result;

    result.x    = e0.x    * t1 + e1.x    * t;
    result.z    = e0.z    * t1 + e1.z    * t;
    result.oow  = e0.oow  * t1 + e1.oow  * t;
    result.fog  = e0.fog  * t1 + e1.fog  * t;
    result.data = e0.data * t1 + e1.data * t;

    return result;
}

unsigned int RasterizeTrapezoid(float f0y, float f1y, EdgePoint const& e00, EdgePoint const& e10, EdgePoint const& e01, EdgePoint const& e11, RasterizeParams const& params, Pixel* pixels, unsigned int pixelsSize)
{
    int const y0 = max((int)floor(f0y + 0.5f)    , (int)params.scissorY);
    int const y1 = min((int)floor(f1y + 0.5f) - 1, (int)(params.scissorY + params.scissorHeight - 1));

    float const oody = 1.0f / (f1y - f0y);

    unsigned int resultCount = 0;

    for (int y = y0; y <= y1; ++y)
    {
        float const t = (float(y) + 0.5f - f0y) * oody;
        float const t1 = 1 - t;

        EdgePoint const e0 = InterpolateEdgePoint(e00, e10, t, t1);
        EdgePoint const e1 = InterpolateEdgePoint(e01, e11, t, t1);

        resultCount += RasterizeScanLine(y, e0, e1, params, pixels + resultCount, pixelsSize - resultCount);
    }

    return resultCount;
}

unsigned int MyRendererFunctions::Rasterize(ScreenTriangle const& triangle, RasterizeParams const& params, Pixel* pixels, unsigned int pixelsSize) const
{
    LightedVertex const* vertex[3] = { &triangle.vtx[0], &triangle.vtx[1], &triangle.vtx[2] };

    if (vertex[0]->position.y <= vertex[1]->position.y)
    {
        if (vertex[1]->position.y <= vertex[2]->position.y)
        {
            // Already ordered.
        }
        else if (vertex[0]->position.y <= vertex[2]->position.y)
        {
            std::swap(vertex[1], vertex[2]);
        }
        else
        {
            std::swap(vertex[1], vertex[2]);
            std::swap(vertex[0], vertex[1]);
        }
    }
    else
    {
        if (vertex[0]->position.y <= vertex[2]->position.y)
        {
            std::swap(vertex[0], vertex[1]);
        }
        else if (vertex[1]->position.y <= vertex[2]->position.y)
        {
            std::swap(vertex[0], vertex[1]);
            std::swap(vertex[1], vertex[2]);
        }
        else
        {
            std::swap(vertex[0], vertex[2]);
        }
    }

    float const yv0 = vertex[0]->position.y;
    float const yv1 = vertex[1]->position.y;
    float const yv2 = vertex[2]->position.y;
    EdgePoint const ev0 = { vertex[0]->position.x, vertex[0]->position.z, vertex[0]->position.w, vertex[0]->fog * vertex[0]->position.w, vertex[0]->data * vertex[0]->position.w };
    EdgePoint const ev1 = { vertex[1]->position.x, vertex[1]->position.z, vertex[1]->position.w, vertex[1]->fog * vertex[1]->position.w, vertex[1]->data * vertex[1]->position.w };
    EdgePoint const ev2 = { vertex[2]->position.x, vertex[2]->position.z, vertex[2]->position.w, vertex[2]->fog * vertex[2]->position.w, vertex[2]->data * vertex[2]->position.w };

    const float mid = (yv1 - yv0) / (yv2 - yv0);
    const float mid1 = 1 - mid;

    EdgePoint const emid = InterpolateEdgePoint(ev0, ev2, mid, mid1);

    unsigned int resultCount = 0;

    if (emid.x < ev1.x)
    {
        resultCount += RasterizeTrapezoid(yv0, yv1, ev0, emid, ev0, ev1, params, pixels, pixelsSize);
        resultCount += RasterizeTrapezoid(yv1, yv2, emid, ev2, ev1, ev2, params, pixels + resultCount, pixelsSize - resultCount);
    }
    else
    {
        resultCount += RasterizeTrapezoid(yv0, yv1, ev0, ev1, ev0, emid, params, pixels, pixelsSize);
        resultCount += RasterizeTrapezoid(yv1, yv2, ev1, ev2, emid, ev2, params, pixels + resultCount, pixelsSize - resultCount);
    }

    return resultCount;
}
